/*********************************************************************
 *
 *                  Master SPI routintes
 *
 *********************************************************************
 * FileName:        MSPI.c
 * Dependencies:	MSPI.h
 * Processor:       PIC18/PIC24
 * Complier:        MCC18 v1.00.50 or higher, C30 3.10 or higher
 *                  HITECH PICC-18 V8.10PL1 or higher
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the Company) for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 ********************************************************************/
#include "MSPI.h"
//SPI setup for PIC24 differs from that of PIC18 processor, so separate routines are written for these cases

#if defined(__PIC24F__)
//SPI read(get) and write(put) routines for PIC24
/************************ FUNCTIONS ********************************/

/*********************************************************************
* Function:         void SPIPut(BYTE v)
*
* PreCondition:     SPI has been configured 
*
* Input:		    v - is the byte that needs to be transfered
*
* Output:		    none
*
* Side Effects:	    SPI transmits the byte
*
* Overview:		    This function will send a byte over the SPI
*
* Note:			    None
********************************************************************/
void SPIPut(BYTE v)
{
    BYTE dummy;
    
		//if not a PIC32 MCU
        IFS0bits.SPI1IF = 0;
		//Clear the SPIxIF bit in the respective IFSx register
        dummy = SPI1BUF;
		//read the value before writing to avoid overflow bit setting
        SPI1BUF = v;
		//write in to buffer the new value
        while(IFS0bits.SPI1IF == 0){}
		//wait until interrupt is set
}

/*********************************************************************
* Function:         BYTE SPIGet(void)
*
* PreCondition:     SPI has been configured 
*
* Input:		    none
*
* Output:		    BYTE - the byte that was last received by the SPI
*
* Side Effects:	    none
*
* Overview:		    This function will read a byte over the SPI
*
* Note:			    None
********************************************************************/
BYTE SPIGet(void)
{

	//for other PIC processors (PIC24, dsPIC33, dsPIC30 etc)
        SPIPut(0x00);
	//put dummy value for transmit and wait until transmit is complete
        return SPI1BUF;
	//return the SPI1BUF value to the calling function (reads the data from the other SPI device)

}

#else
//for PIC18s
void SPIPut(BYTE v)
{
	INTCONbits.GIE = 0;	
		//Reset the Global interrupt pin
    SSPIF = 0;
    do
    {
        WCOL = 0;
		//Reset write collision bit
        SSPBUF = v;
		//load the buffer
    } while( WCOL );
		//perform write again if write collision occurs
    while( SSPIF == 0 );
		//Wait until interrupt is received from the MSSP module
	SSPIF = 0;
		//Reset the interrupt
	INTCONbits.GIE = 1;
		//Enable the global interrupt pin
}

BYTE SPIGet(void)
{
    SPIPut(0x00);
		//write a dummy value to read the data from the other SPI module
    return SSPBUF;
		//read the SSPBUF value for data transmitted by other SSP module
}
#endif




